<?php

/*
 * This file is part of the Dektrium project.
 *
 * (c) Dektrium project <http://github.com/dektrium/>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace dektrium\user\controllers;

use dektrium\user\Finder;
use dektrium\user\models\RegistrationForm;
use dektrium\user\models\ResendForm;
use dektrium\user\models\User;
use dektrium\user\traits\AjaxValidationTrait;
use dektrium\user\traits\EventTrait;
use yii\filters\AccessControl;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\db\ActiveRecord;
use yii;

/**
 * RegistrationController is responsible for all registration process, which includes registration of a new account,
 * resending confirmation tokens, email confirmation and registration via social networks.
 *
 * @property \dektrium\user\Module $module
 *
 * @author Dmitry Erofeev <dmeroff@gmail.com>
 */
class RegistrationController extends Controller
{
    use AjaxValidationTrait;
    use EventTrait;

    /**
     * Event is triggered after creating RegistrationForm class.
     * Triggered with \dektrium\user\events\FormEvent.
     */
    const EVENT_BEFORE_REGISTER = 'beforeRegister';

    /**
     * Event is triggered after successful registration.
     * Triggered with \dektrium\user\events\FormEvent.
     */
    const EVENT_AFTER_REGISTER = 'afterRegister';

    /**
     * Event is triggered before connecting user to social account.
     * Triggered with \dektrium\user\events\UserEvent.
     */
    const EVENT_BEFORE_CONNECT = 'beforeConnect';

    /**
     * Event is triggered after connecting user to social account.
     * Triggered with \dektrium\user\events\UserEvent.
     */
    const EVENT_AFTER_CONNECT = 'afterConnect';

    /**
     * Event is triggered before confirming user.
     * Triggered with \dektrium\user\events\UserEvent.
     */
    const EVENT_BEFORE_CONFIRM = 'beforeConfirm';

    /**
     * Event is triggered before confirming user.
     * Triggered with \dektrium\user\events\UserEvent.
     */
    const EVENT_AFTER_CONFIRM = 'afterConfirm';

    /**
     * Event is triggered after creating ResendForm class.
     * Triggered with \dektrium\user\events\FormEvent.
     */
    const EVENT_BEFORE_RESEND = 'beforeResend';

    /**
     * Event is triggered after successful resending of confirmation email.
     * Triggered with \dektrium\user\events\FormEvent.
     */
    const EVENT_AFTER_RESEND = 'afterResend';

    /** @var Finder */
    protected $finder;

    /**
     * @param string           $id
     * @param \yii\base\Module $module
     * @param Finder           $finder
     * @param array            $config
     */
    public function __construct($id, $module, Finder $finder, $config = [])
    {
        $this->finder = $finder;
        parent::__construct($id, $module, $config);
    }

    /** @inheritdoc */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    ['allow' => true, 'actions' => ['register', 'connect'], 'roles' => ['?']],
                    ['allow' => true, 'actions' => ['confirm', 'resend'], 'roles' => ['?', '@']],
                ],
            ],
        ];
    }

    /**
     * Displays the registration page.
     * After successful registration if enableConfirmation is enabled shows info message otherwise
     * redirects to home page.
     *
     * @return string
     * @throws \yii\web\HttpException
     */    
    public function actionRegister()
    {
		if(isset($_SESSION['userid']))
		{
			$url = \Yii::$app->request->baseUrl;
            return $this->redirect($url);
		}
		
		if (!$this->module->enableRegistration) {
            throw new NotFoundHttpException();
        }

        /** @var RegistrationForm $model */
        $model = \Yii::createObject(RegistrationForm::className());

        $event = $this->getFormEvent($model);

        $this->trigger(self::EVENT_BEFORE_REGISTER, $event);

        $this->performAjaxValidation($model);


        if ($model->load(\Yii::$app->request->post())) {
			$recaptcha_secret = "0x4AAAAAAAOEsBV8JvEK9lCZVRsxuzKQno8";  // Replace with your actual secret key   
			$recaptcha_response = $_POST['g-recaptcha-response'] ?? '';

			if (!$recaptcha_response) {
				die("reCAPTCHA verification failed. Please try again.");
			}
			if(!empty($recaptcha_response) && $recaptcha_response!="bypass"){
				$ip = $_SERVER['REMOTE_ADDR']; 
				$url_path = 'https://challenges.cloudflare.com/turnstile/v0/siteverify';
				$data = array('secret' => $recaptcha_secret, 'response' => $recaptcha_response, 'remoteip' => $ip);

				$options = array(
					'http' => array(
					'header' => "Content-type: application/x-www-form-urlencoded", 
					'method' => 'POST',
					'content' => http_build_query($data))
				);
				
				$stream = stream_context_create($options);

				$result = file_get_contents($url_path, false, $stream);

				$response =  $result;   

				$recaptcha_data = json_decode($response,true);
				//print_r($recaptcha_data);exit; 
				if (!$recaptcha_data['success']) { 
					//return false; 
					return $this->render('registerlive'); 
				}
			}
			
			$model->register();
            $this->trigger(self::EVENT_AFTER_REGISTER, $event);

            // Set teamid=userid here
            if(\Yii::$app->request->post()['register-form']['email']){               
                $mymodel = new User();
                $mymodel->setMyTeamId(\Yii::$app->request->post()['register-form']['email'],\Yii::$app->request->post()['register-form']['portal_id']);   
            } 
			if (!empty(\Yii::$app->request->post('partnero_partner'))){
				// Do something with the cookie value
				/* echo 'Cookie value: ' . $_COOKIE['partnero_session_uuid'];                   
				echo 'Cookie partnero_partner value: ' . $_COOKIE['partnero_partner']; 
				echo 'userId : ' .$mymodel->id;exit; */                    
				$url = 'https://api.partnero.com/v1/customers';
				$partnerKey = \Yii::$app->request->post('partnero_partner'); 
				$customerKey = Yii::$app->db->createCommand("select id from  user  where email ='".$model->email."'")->queryScalar(); 

				$data = [
					'partner' => [
						'key' => $partnerKey,
					],
					'key' => $customerKey,
					'name' => $model->first_name,
					'surname' => $model->last_name,
					'email'=> $model->email,
				];
				$headers = [
					'Accept: application/json',
					'Content-Type: application/json',
					'Authorization: Bearer eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJhdWQiOiI1IiwianRpIjoiMjc5NjdhNTgyOTkyMGRkYzU5OWQyZTI2YWI0MDdmNzIzN2EwZjQ4MWQ5MWI3ZDYyNDQxYmI3ZGE2YWFjNWFmNTdlYzE4NDVkZmI2ZDdhZWEiLCJpYXQiOjE3MDMyNzcwMzguNTY3MDY1LCJuYmYiOjE3MDMyNzcwMzguNTY3MDY4LCJleHAiOjQ4NTg5NTA2MzguNTU5ODQxLCJzdWIiOiIzMDcyIiwic2NvcGVzIjpbXX0.VAeZI0RpAQlHl7mY-hii6YxwnpbSmABP1TzzJRcXJxWhxHuNMN9-0aJD8py3FD-_w9kOTOpWMwvRpMaCI93QrhTUMOl1C1O0u8tBt0PPdvwPa0zqgbPnLTAEUN7Wjhdqxv3Q8bjF7Kl48eAG326dTfJxt4zgcGot3Lr4SfTfqw6RSJMD4S-uBHGYqPdo-O312gCfjeis63n3ji071j7SNMJDN8blz2olKf6e-kKjsW0a_SqOI6Jkvw7FKNeoyECnKkwwrprBQ3J2qTetCqQjZ4IsLk0twHvo3MDUNNhvxH3YzAxh8CtbUzr__BzlAgNc6sp130LLmXD0NRlpDwTN6CSNtCO43SH9TRk_aJ8leHWy10nKp47l_5l8QMzkOaOFchBnBfx5VTKUk9DG5b1y1dTMRHZI0zYoXmBljzNMes2sFiSpeZev5Iy-SI67t8nMF8395jsDPHGMpAP6yXYdWg5DpO1eTYYL5hoKSdyQFs1UBWUtjHMDXpIpH3fcjA4uKAYm2YfLeM8TDHB9oZfDo2EiANhSgbv39uREsM1azSrmTQQ3ERaZm-Bw8T6YCnIiSGbtjriXNAtrgTYni_a_Z0C5d6FtmpfL2eaHBx3J5k0v9MJ6_F6jDYwWIkxgqUpZ1tgT5oUU3Mi095fK0dawhTnHnm45x1ImJW0_RvrHZyI'
				]; 

				$ch = curl_init($url);
				curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
				curl_setopt($ch, CURLOPT_POST, true);
				curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
				curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
				$response = curl_exec($ch);
				if (curl_errno($ch)) {
					echo 'Curl error: ' . curl_error($ch);
				}
				curl_close($ch);
			}

			return $this->redirect('login');			
        }
		
		if(isset($_REQUEST['test']) && $_REQUEST['test'] == 1){
			return $this->render('registerlive', [
				'model'  => $model,
				'module' => $this->module,
			]);	
		}else{
			return $this->render('registerlive', [
				'model'  => $model,
				'module' => $this->module,
			]);
		}
    }


    /**
     * Displays page where user can create new account that will be connected to social account.
     *
     * @param string $code
     *
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionConnect($code)
    {
        $account = $this->finder->findAccount()->byCode($code)->one();

        if ($account === null || $account->getIsConnected()) {
            throw new NotFoundHttpException();
        }

        /** @var User $user */
        $user = \Yii::createObject([
            'class'    => User::className(),
            'scenario' => 'connect',
            'username' => $account->username,
            'email'    => $account->email,
        ]);

        $event = $this->getConnectEvent($account, $user);

        $this->trigger(self::EVENT_BEFORE_CONNECT, $event);

        if ($user->load(\Yii::$app->request->post()) && $user->create()) {
            $account->connect($user);
            $this->trigger(self::EVENT_AFTER_CONNECT, $event);
            \Yii::$app->user->login($user, $this->module->rememberFor);
            return $this->goBack();
        }

        return $this->render('connect', [
            'model'   => $user,
            'account' => $account,
        ]);
    }

    /**
     * Confirms user's account. If confirmation was successful logs the user and shows success message. Otherwise
     * shows error message.
     *
     * @param int    $id
     * @param string $code
     *
     * @return string
     * @throws \yii\web\HttpException
     */
    public function actionConfirm($id, $code)
    {
        $user = $this->finder->findUserById($id);

        if ($user === null || $this->module->enableConfirmation == false) {
            throw new NotFoundHttpException();
        }

        $event = $this->getUserEvent($user);

        $this->trigger(self::EVENT_BEFORE_CONFIRM, $event);

        $user->attemptConfirmation($code);

        $this->trigger(self::EVENT_AFTER_CONFIRM, $event);

        return $this->render('/message', [
            'title'  => \Yii::t('user', 'Account confirmation'),
            'module' => $this->module,
        ]);
    }

    /**
     * Displays page where user can request new confirmation token. If resending was successful, displays message.
     *
     * @return string
     * @throws \yii\web\HttpException
     */
    public function actionResend()
    {
        if ($this->module->enableConfirmation == false) {
            throw new NotFoundHttpException();
        }

        /** @var ResendForm $model */
        $model = \Yii::createObject(ResendForm::className());
        $event = $this->getFormEvent($model);

        $this->trigger(self::EVENT_BEFORE_RESEND, $event);

        $this->performAjaxValidation($model);

        if ($model->load(\Yii::$app->request->post()) && $model->resend()) {
            $this->trigger(self::EVENT_AFTER_RESEND, $event);
			// This is commented to stay on same page
            /*return $this->render('/message', [
                'title'  => \Yii::t('user', 'A new confirmation link has been sent'),
                'module' => $this->module,
            ]);*/
        }

        return $this->render('resend', [
            'model' => $model,
        ]);
    }
}
